<?php
session_start();
require_once 'vendor/autoload.php'; // For mpdf
require_once 'config.php';
require_once 'includes/emission_factors_config.php'; // CRITICAL for EFs

if (!isset($_SESSION['general_id'])) {
    die("No data available to download. Please generate a summary first.");
}
$general_id = $_SESSION['general_id'];

// --- Retrieve dynamic chart filenames from GET parameters ---
// Ensure these keys match EXACTLY what is sent from the JavaScript
$chart_param_keys = [
    'ghg_by_scope' => 'chart_ghg_by_scope',
    'stationary_breakdown' => 'chart_stationary_breakdown',
    'mobile_breakdown' => 'chart_mobile_breakdown',
    'fugitive_breakdown' => 'chart_fugitive_breakdown'
];

$chart_filenames = [];
foreach ($chart_param_keys as $key_internal => $param_name_from_get) {
    // Sanitize by allowing only alphanumeric, underscore, hyphen, dot.
    // This is important as GET params can be manipulated.
    $filename_from_get = $_GET[$param_name_from_get] ?? 'default_' . $key_internal . '.png';
    $chart_filenames[$key_internal] = preg_replace('/[^A-Za-z0-9_.-]/', '', $filename_from_get);
}

// --- Retrieve dynamic chart filenames from GET parameters ---
$chart_param_keys_and_defaults = [
    'ghg_by_scope' => ['param_get_key' => 'chart_ghg_by_scope', 'alt_text' => 'GHG By Scope Chart', 'no_data_message' => 'GHG By Scope Chart - No Data to Display'],
    'stationary_breakdown' => ['param_get_key' => 'chart_stationary_breakdown', 'alt_text' => 'Stationary Breakdown', 'no_data_message' => 'Stationary Breakdown - No Data to Display'],
    'mobile_breakdown' => ['param_get_key' => 'chart_mobile_breakdown', 'alt_text' => 'Mobile Breakdown', 'no_data_message' => 'Mobile Breakdown - No Data to Display'],
    'fugitive_breakdown' => ['param_get_key' => 'chart_fugitive_breakdown', 'alt_text' => 'Fugitive Breakdown', 'no_data_message' => 'Fugitive Breakdown - No Data to Display']
];

$chart_image_html = []; // This will store the full HTML (img tag or "No Data" message)

foreach ($chart_param_keys_and_defaults as $internal_key => $info) {
    $get_param_name = $info['param_get_key'];
    $alt_text = $info['alt_text'];
    $no_data_message = $info['no_data_message'];
    // Default to "No Data" message
    $image_html_output = '<p style="text-align:center; border:1px solid #ccc; padding:20px; margin:10px 0;"><em>' . htmlspecialchars($no_data_message) . '</em></p>'; 

    if (isset($_GET[$get_param_name]) && !empty($_GET[$get_param_name])) {
        $filename_from_get = $_GET[$get_param_name];
        $sanitized_filename = preg_replace('/[^A-Za-z0-9_.-]/', '', $filename_from_get);

        if (!empty($sanitized_filename)) {
            // Adjust this path based on your actual directory structure:
            // If download_pdf.php is in root and temp_charts is in root: 'temp_charts/'
            // If download_pdf.php is in admin/ and temp_charts is in root: '../temp_charts/'
            $image_server_path = 'temp_charts/' . $sanitized_filename; // Assuming download_pdf.php is in the root with temp_charts

            if (file_exists($image_server_path) && filesize($image_server_path) > 0) {
                $image_html_output = '<img src="' . htmlspecialchars($image_server_path) . '" alt="' . htmlspecialchars($alt_text) . '" class="chart-image">';
            } else {
                error_log("PDF Generation: Chart image file NOT FOUND or EMPTY at: " . $image_server_path . " (Filename from GET: {$filename_from_get})");
                // $image_html_output remains the "No Data" message
            }
        } else {
             error_log("PDF Generation: Filename became empty after sanitization for GET param: {$get_param_name}");
        }
    } else {
        // Parameter not set for this chart, assume it was skipped due to no data
        // $image_html_output remains the "No Data" message
        error_log("PDF Generation: GET parameter '{$get_param_name}' not set for chart '{$internal_key}'.");
    }
    $chart_image_html[$internal_key] = $image_html_output;
}

// --- START: DATA FETCHING & AGGREGATION LOGIC (COPY FROM WORKING download_excel.php or output.php) ---
$years = [2021, 2022, 2023, 2024];
$categories_summary = [
    'total_scope1' => array_fill_keys($years, 0.0),
    'total_scope2' => array_fill_keys($years, 0.0),
    'total_scope1_plus_2' => array_fill_keys($years, 0.0),
    's1_stationary' => array_fill_keys($years, 0.0),
    's1_mobile' => array_fill_keys($years, 0.0),
    's1_fugitive' => array_fill_keys($years, 0.0),
    's2_electricity' => array_fill_keys($years, 0.0),
    's2_other_energy' => array_fill_keys($years, 0.0),
];
// Note: You also need $s1_stationary_breakdown etc. if you want to pass their data
// to the HTML part that might be used for chart image generation placeholders later.

$scope1_emission_id = null;
$s1_main_db_totals = null;
$sql_s1_main = "SELECT id FROM scope1_emissions WHERE general_id = ? LIMIT 1"; // Only need ID to link details
$stmt_s1_main = mysqli_prepare($conn, $sql_s1_main);

if ($stmt_s1_main) {
    mysqli_stmt_bind_param($stmt_s1_main, "i", $general_id);
    if (mysqli_stmt_execute($stmt_s1_main)) {
        $res_s1_main = mysqli_stmt_get_result($stmt_s1_main);
        if ($res_s1_main) {
            $s1_main_db_totals = mysqli_fetch_assoc($res_s1_main);
            if ($s1_main_db_totals) {
                $scope1_emission_id = $s1_main_db_totals['id'];
            }
        } else { /* Log error getting result */ }
    } else { /* Log error executing */ }
    mysqli_stmt_close($stmt_s1_main);
} else { /* Log error preparing */ }

if ($scope1_emission_id) {
    // Stationary Combustion Details
    $stmt_stat = mysqli_prepare($conn, "SELECT fuel_type, input_unit, quantity_2021, quantity_2022, quantity_2023, quantity_2024 FROM stationary_combustion_details WHERE scope1_emission_id = ?");
    if($stmt_stat) {
        mysqli_stmt_bind_param($stmt_stat, "i", $scope1_emission_id); 
		mysqli_stmt_execute($stmt_stat);
        $result_stat = mysqli_stmt_get_result($stmt_stat);
        while ($row = mysqli_fetch_assoc($result_stat)) {
            $ef = $emission_factors_data['stationary'][$row['fuel_type']][$row['input_unit']] ?? 0;
            if ($ef > 0) { 
				foreach ($years as $year) { 
					$qty_col = 'quantity_' . $year;
					$co2e_year = (floatval($row[$qty_col] ?? 0) * $ef);
					$categories_summary['s1_stationary'][$year] += $co2e_year;
					$categories_summary['total_scope1'][$year] += $co2e_year;
				} 
			}
        }
		mysqli_stmt_close($stmt_stat);
    } // else { error_log("Error preparing S1 Stationary details: ".mysqli_error($conn)); }

    // Mobile Combustion Details
    $stmt_mob = mysqli_prepare($conn, "SELECT fuel_type, input_unit, quantity_2021, quantity_2022, quantity_2023, quantity_2024 FROM mobile_combustion_details WHERE scope1_emission_id = ?");
    if($stmt_mob){
        mysqli_stmt_bind_param($stmt_mob, "i", $scope1_emission_id); 
		mysqli_stmt_execute($stmt_mob);
        $result_mob = mysqli_stmt_get_result($stmt_mob);
        while ($row = mysqli_fetch_assoc($result_mob)) {
            $ef = $emission_factors_data['mobile'][$row['fuel_type']][$row['input_unit']] ?? 0;
            if ($ef > 0) {
				foreach ($years as $year) {
					$qty_col = 'quantity_' . $year; 
					$co2e_year = (floatval($row[$qty_col] ?? 0) * $ef); 
					$categories_summary['s1_mobile'][$year] += $co2e_year; 
					$categories_summary['total_scope1'][$year] += $co2e_year; 
				} 
			}
        } 
		mysqli_stmt_close($stmt_mob);
    } // else { error_log("Error preparing S1 Mobile details: ".mysqli_error($conn)); }

    // Fugitive Emissions Details
    $stmt_fug = mysqli_prepare($conn, "SELECT gas_type, input_unit, quantity_2021, quantity_2022, quantity_2023, quantity_2024 FROM fugitive_emission_details WHERE scope1_emission_id = ?");
    if($stmt_fug){
        mysqli_stmt_bind_param($stmt_fug, "i", $scope1_emission_id); 
		mysqli_stmt_execute($stmt_fug);
        $result_fug = mysqli_stmt_get_result($stmt_fug);
        while ($row = mysqli_fetch_assoc($result_fug)) {
            $gwp = $emission_factors_data['fugitive'][$row['gas_type']][$row['input_unit']] ?? $emission_factors_data['fugitive'][$row['gas_type']]['kg'] ?? 0;
            if ($gwp > 0) { 
				foreach ($years as $year) { 
					$qty_col = 'quantity_' . $year; 
					$qty_original_unit = floatval($row[$qty_col] ?? 0); 
					$qty_in_tonnes = ($row['input_unit'] === 'kg') ? $qty_original_unit : (($row['input_unit'] === 'metric tonnes') ? $qty_original_unit : 0) ; 
					if ($qty_in_tonnes > 0) { 
						$co2e_year = $qty_in_tonnes * $gwp; 
						$categories_summary['s1_fugitive'][$year] += $co2e_year; 
						$categories_summary['total_scope1'][$year] += $co2e_year; 
					}
				} 
			}
        } 
		mysqli_stmt_close($stmt_fug);
    }
}

$scope2_emission_id = null;
$s2_main_db_data = null; // Renamed to avoid conflict if you had $s2_main_db_totals
$sql_s2_main = "SELECT id, include_steam_heat_cooling FROM scope2_emissions WHERE general_id = ? LIMIT 1";
$stmt_s2_main = mysqli_prepare($conn, $sql_s2_main);
if($stmt_s2_main) {
    mysqli_stmt_bind_param($stmt_s2_main, "i", $general_id);
    if(mysqli_stmt_execute($stmt_s2_main)) {
        $res_s2_main = mysqli_stmt_get_result($stmt_s2_main);
        if($res_s2_main) {
            $s2_main_db_data = mysqli_fetch_assoc($res_s2_main);
            if($s2_main_db_data) {
                $scope2_emission_id = $s2_main_db_data['id'];
            }
        }
    }
    mysqli_stmt_close($stmt_s2_main);
}

if ($scope2_emission_id) {
    // Electricity Consumption Details
    $stmt_elec = mysqli_prepare($conn, "SELECT region, input_unit, quantity_2021, quantity_2022, quantity_2023, quantity_2024 FROM electricity_consumption_details WHERE scope2_emission_id = ?");
    if($stmt_elec){
        mysqli_stmt_bind_param($stmt_elec, "i", $scope2_emission_id); 
        mysqli_stmt_execute($stmt_elec);
        $result_elec = mysqli_stmt_get_result($stmt_elec);
        
        while ($row = mysqli_fetch_assoc($result_elec)) { // $row contains data for the current electricity entry
            // $raw_s2_electricity[] = $row; // If you also need to store raw data for PDF display

            // Get region and input_unit FROM THE CURRENT $row
            $current_row_region = $row['region'];
            $current_row_input_unit = $row['input_unit'];
			$year_column_keys = ['col1', 'col2', 'col3', 'col4'];

            // Iterate through each year to apply the year-specific EF
            foreach (array_keys($years) as $year_index) { // This $year is 2021, then 2022, etc.
                $current_year = $years[$year_index]; // Get 2021, 2022, etc.
                $ef_key_for_year = $year_column_keys[$year_index]; // Get col1, col2, etc.

                $qty_col = 'quantity_' . $current_year;
                $quantity_value = floatval($row[$qty_col] ?? 0);

                if ($quantity_value > 0) {
                    $ef_for_year = 0;
                    if (isset($emission_factors_data_scope2['electricity'][$current_row_region]) && isset($emission_factors_data_scope2['electricity'][$current_row_region][$current_row_input_unit]) && isset($emission_factors_data_scope2['electricity'][$current_row_region][$current_row_input_unit][$ef_key_for_year])) {
                        $ef_for_year = $emission_factors_data_scope2['electricity'][$current_row_region][$current_row_input_unit][$ef_key_for_year];
                    } // else { error_log(...); }
                    
                    if ($ef_for_year > 0) {
                        $co2e_year = ($quantity_value * $ef_for_year);
                        // Use $current_year to store in $categories_summary
                        $categories_summary['s2_electricity'][$current_year] += $co2e_year;
                        $categories_summary['total_scope2'][$current_year] += $co2e_year;
                    }
                }
            }
		} 
		mysqli_stmt_close($stmt_elec);
    } else {
        error_log("PDF Download - Error preparing S2 Electricity details: ".mysqli_error($conn));
	}

    // Steam, Heat & Cooling Details
    if ($s2_main_db_data && $s2_main_db_data['include_steam_heat_cooling']) {
        $stmt_shc = mysqli_prepare($conn, "SELECT energy_type, quantity_2021, quantity_2022, quantity_2023, quantity_2024 FROM steam_heat_cooling_details WHERE scope2_emission_id = ?");
        if($stmt_shc){
            mysqli_stmt_bind_param($stmt_shc, "i", $scope2_emission_id); mysqli_stmt_execute($stmt_shc);
            $result_shc = mysqli_stmt_get_result($stmt_shc);
            while ($row = mysqli_fetch_assoc($result_shc)) {
                $ef = $emission_factors_data_scope2['steam_heat_cooling'][$row['energy_type']] ?? 0;
                if ($ef > 0) { 
					foreach ($years as $year) { 
						$qty_col = 'quantity_' . $year; 
						$co2e_year = (floatval($row[$qty_col] ?? 0) * $ef); 
						$categories_summary['s2_other_energy'][$year] += $co2e_year; 
						$categories_summary['total_scope2'][$year] += $co2e_year; 
					} 
				}
            } 
			mysqli_stmt_close($stmt_shc);
        }
    }
}

// --- NEW: Fetch Annual Revenue Data ---
$annual_revenues_by_year = array_fill_keys($years, null); // Initialize with nulls
$stmt_revenue = mysqli_prepare($conn, "SELECT revenue_2021, revenue_2022, revenue_2023, revenue_2024 FROM annual_revenues WHERE general_id = ? LIMIT 1");
if ($stmt_revenue) {
    mysqli_stmt_bind_param($stmt_revenue, "i", $general_id);
    if (mysqli_stmt_execute($stmt_revenue)) {
        $result_revenue = mysqli_stmt_get_result($stmt_revenue);
        if ($revenue_data_row = mysqli_fetch_assoc($result_revenue)) {
            foreach ($years as $year) {
                $annual_revenues_by_year[$year] = isset($revenue_data_row['revenue_'.$year]) ? floatval($revenue_data_row['revenue_'.$year]) : null;
            }
        }
    }
    mysqli_stmt_close($stmt_revenue);
}

// --- NEW: Initialize and Calculate Intensity Metrics ---
$intensity_metrics = [
    'scope1_intensity' => array_fill_keys($years, null), // Use null for 'N/A' or if calculation is not possible
    'scope2_intensity' => array_fill_keys($years, null),
    'total_intensity' => array_fill_keys($years, null),
];
// --- NEW: Fetch Annual Revenue Data ---

// Calculate Total Scope 1 + 2
foreach ($years as $year) {
	$categories_summary['total_scope1_plus_2'][$year] = ($categories['total_scope1'][$year] ?? 0.0) + ($categories_summary['total_scope2'][$year] ?? 0.0);
	
    $total_s1 = $categories_summary['total_scope1'][$year] ?? 0.0;
    $total_s2 = $categories_summary['total_scope2'][$year] ?? 0.0;
    $revenue_for_this_year = $annual_revenues_by_year[$year] ?? null;

    if ($revenue_for_this_year !== null && $revenue_for_this_year > 0) {
        $divisor_revenue = $revenue_for_this_year / 1000000.0; // For "Tonnes CO2e/million RM revenue"
        if ($divisor_revenue > 0) { 
            $intensity_metrics['scope1_intensity'][$year] = $total_s1 / $divisor_revenue;
            $intensity_metrics['scope2_intensity'][$year] = $total_s2 / $divisor_revenue;
            $intensity_metrics['total_intensity'][$year] = ($total_s1 + $total_s2) / $divisor_revenue;
        }
    }
}
// --- END: DATA FETCHING & AGGREGATION LOGIC ---

// --- Start Output Buffering to Capture HTML ---
ob_start();
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <title>GHG Summary Report - PDF</title>
    <style>
        body { font-family: sans-serif; font-size: 10pt; }
        .summary-table { width: 100%; border-collapse: collapse; margin-bottom: 20px; font-size: 9pt;}
        .summary-table th, .summary-table td { border: 1px solid #666; padding: 5px; text-align: left; }
        .summary-table th { background-color: #f0f0f0; font-weight: bold; }
        .summary-table td.year-data { text-align: right; }
        .section-title { font-size: 1.2em; margin-top: 15px; margin-bottom: 8px; border-bottom: 1px solid #ccc; padding-bottom: 3px; font-weight:bold;}
        h1 {text-align: center; margin-bottom: 20px;}
        /* Styles for charts if you embed them as images */
        .chart-image { max-width: 100%; height: auto; margin-bottom: 20px; border: 1px solid #ccc; }
        .chart-group { margin-bottom: 20px; }
        .chart-container-pdf { width: 100%; display: inline-block; margin-right: 2%; vertical-align: top; } /* For side-by-side charts */
        .chart-container-pdf.last { margin-right: 0; }
        .page-break { page-break-before: always; }
    </style>
</head>
<body>
    <h1>GHG Emissions Summary Report</h1>
    <p>Report for General ID: <?php echo htmlspecialchars($general_id); ?> (Generated: <?php echo date("Y-m-d H:i:s"); ?>)</p>

    <div class="section-title">Total GHG Emissions</div>
    <table class="summary-table">
        <thead>
			<tr>
				<th>Category</th>
				<th>Unit</th><?php foreach ($years as $y) echo "<th>{$y}</th>"; ?>
			</tr>
		</thead>
        <tbody>
            <tr>
				<td>Total Scope 1</td><td>metric tonnes of CO2e</td><?php foreach ($years as $y) echo "<td class='year-data'>".number_format($categories_summary['total_scope1'][$y] ?? 0, 2)."</td>"; ?>
			</tr>
            <tr>
				<td>Total Scope 2</td><td>metric tonnes of CO2e</td><?php foreach ($years as $y) echo "<td class='year-data'>".number_format($categories_summary['total_scope2'][$y] ?? 0, 2)."</td>"; ?>
			</tr>
            <tr style="font-weight: bold;">
				<td>Total Scope 1+2</td><td>metric tonnes of CO2e</td><?php foreach ($years as $y) echo "<td class='year-data'>".number_format($categories_summary['total_scope1_plus_2'][$y] ?? 0, 2)."</td>"; ?>
			</tr>
        </tbody>
    </table>
	
	<div class="section-title">Total Intensity</div>
    <table class="summary-table">
        <thead>
			<tr>
				<th>Category</th>
				<th>Unit</th><?php foreach ($years as $y) echo "<th>{$y}</th>"; ?>
			</tr>
		</thead>
        <tbody>
            <?php $intensityUnitPDF = 'metric tonnes CO2e/million RM revenue'; ?>
            <tr>
                <td>Scope 1 Intensity</td>
                <td><?php echo $intensityUnitPDF; ?></td>
                <?php foreach ($years as $y): ?>
                    <td class='year-data'>
                        <?php echo ($intensity_metrics['scope1_intensity'][$y] !== null) ? htmlspecialchars(number_format($intensity_metrics['scope1_intensity'][$y], 2)) : '-'; ?>
                    </td>
                <?php endforeach; ?>
            </tr>
            <tr>
                <td>Scope 2 Intensity</td>
                <td><?php echo $intensityUnitPDF; ?></td>
                 <?php foreach ($years as $y): ?>
                    <td class='year-data'>
                        <?php echo ($intensity_metrics['scope2_intensity'][$y] !== null) ? htmlspecialchars(number_format($intensity_metrics['scope2_intensity'][$y], 2)) : '-'; ?>
                    </td>
                <?php endforeach; ?>
            </tr>
            <tr style="font-weight: bold;">
                <td>Total Intensity (Scope 1+2)</td>
                <td><?php echo $intensityUnitPDF; ?></td>
                 <?php foreach ($years as $y): ?>
                    <td class='year-data'>
                        <?php echo ($intensity_metrics['total_intensity'][$y] !== null) ? htmlspecialchars(number_format($intensity_metrics['total_intensity'][$y], 2)) : '-'; ?>
                    </td>
                <?php endforeach; ?>
            </tr>
        </tbody>
    </table>

    <div class="section-title">Scope 1 Breakdown</div>
    <table class="summary-table">
        <thead>
			<tr>
				<th>Category</th>
				<th>Unit</th><?php foreach ($years as $y) echo "<th>{$y}</th>"; ?>
			</tr>
		</thead>
        <tbody>
            <tr>
				<td>Stationary Combustion</td><td>metric tonnes of CO2e</td><?php foreach ($years as $y) echo "<td class='year-data'>".number_format($categories_summary['s1_stationary'][$y] ?? 0, 2)."</td>"; ?>
			</tr>
            <tr>
				<td>Mobile Combustion</td><td>metric tonnes of CO2e</td><?php foreach ($years as $y) echo "<td class='year-data'>".number_format($categories_summary['s1_mobile'][$y] ?? 0, 2)."</td>"; ?>
			</tr>
            <tr>
				<td>Fugitive Emissions (Refrigerants)</td><td>metric tonnes of CO2e</td><?php foreach ($years as $y) echo "<td class='year-data'>".number_format($categories_summary['s1_fugitive'][$y] ?? 0, 2)."</td>"; ?>
			</tr>
        </tbody>
    </table>

    <div class="section-title">Scope 2 Breakdown</div>
    <table class="summary-table">
        <thead>
			<tr>
				<th>Category</th>
				<th>Unit</th><?php foreach ($years as $y) echo "<th>{$y}</th>"; ?>
			</tr>
		</thead>
        <tbody>
            <tr>
				<td>Electricity Consumption</td>
				<td>metric tonnes of CO2e</td><?php foreach ($years as $y) echo "<td class='year-data'>".number_format($categories_summary['s2_electricity'][$y] ?? 0, 2)."</td>"; ?>
			</tr>
        </tbody>
    </table>

    <!-- Placeholder for Charts - See Step 2.3 for how to include them -->
    <!-- You might need to save charts as images on the server first, then embed them -->
    <!-- Or use a library that can directly render Chart.js in PDF (more advanced) -->
    <div class="section-title page-break">Charts (Images)</div>
    <p><em>Note: Charts are embedded as images. For dynamic charts, view the web summary.</em></p>
    
    <div><?php echo $chart_image_html['ghg_by_scope']; ?></div>
    
    <div class="chart-group">
        <div class="chart-container-pdf"><?php echo $chart_image_html['stationary_breakdown']; ?></div>
        <div class="chart-container-pdf last"><?php echo $chart_image_html['mobile_breakdown']; ?></div>
    </div>
</body>
</html>

<?php
$html = ob_get_clean(); // Get HTML content from buffer

try {
    $mpdf = new \Mpdf\Mpdf([
        'mode' => 'utf-8',
        'format' => 'A4-P', // A4 Portrait
        'margin_left' => 10,
        'margin_right' => 10,
        'margin_top' => 15,
        'margin_bottom' => 15,
    ]);

    // For complex CSS or external stylesheets, you might need to load them explicitly
    // $stylesheet = file_get_contents('path/to/your_pdf_styles.css');
    // $mpdf->WriteHTML($stylesheet, \Mpdf\HTMLParserMode::HEADER_CSS);

    $mpdf->WriteHTML($html);
    $filename = "ghg_summary_report_" . $general_id . "_" . date('YmdHis') . ".pdf"; // Add timestamp to PDF filename too
    $mpdf->Output($filename, 'D'); // 'D' for download, 'I' for inline

} catch (\Mpdf\MpdfException $e) {
    echo "MPDF Error: " . $e->getMessage();
}
exit;
?>